@extends('layouts.app')

@section('content')
<!-- Hero Section with Breadcrumb - Home Page Theme -->
<section class="relative min-h-[40vh] flex items-center overflow-hidden bg-gradient-to-br from-[#3d6320] via-[#558B2F] to-[#4a7a28]">
    <!-- Animated Background Pattern -->
    <div class="absolute inset-0 opacity-10">
        <div class="absolute inset-0" style="background-image: radial-gradient(circle at 2px 2px, rgba(255,255,255,0.15) 1px, transparent 0); background-size: 40px 40px;"></div>
    </div>
    
    <!-- Floating Particles -->
    <div class="absolute inset-0 overflow-hidden pointer-events-none">
        <div class="particle-float absolute w-2 h-2 bg-white/30 rounded-full" style="top: 20%; left: 15%; animation-delay: 0s;"></div>
        <div class="particle-float absolute w-3 h-3 bg-[#8bc34a]/30 rounded-full" style="top: 60%; left: 75%; animation-delay: 2s;"></div>
    </div>

    <div class="container mx-auto px-4 relative z-10 py-20">
        <div class="max-w-4xl mx-auto text-center">
            <!-- Back Button -->
            <div class="mb-8">
                <a href="{{ route('rooms.index') }}" class="inline-flex items-center gap-2 px-6 py-3 bg-white/10 backdrop-blur-sm border border-white/20 text-white rounded-md hover:bg-white/20 transition-all duration-300 group uppercase tracking-widest text-sm font-bold">
                    <svg class="w-5 h-5 group-hover:-translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 19l-7-7m0 0l7-7m-7 7h18"/>
                    </svg>
                    Back to Rooms
                </a>
            </div>

            <h1 class="text-4xl md:text-6xl font-display font-bold text-white leading-tight mb-4 animate-fade-in-up">
                {{ $room->name }}
            </h1>
            <p class="text-white/90 text-xl animate-fade-in-up" style="animation-delay: 0.2s;">
                {{ $room->type }} Room • {{ $room->capacity }} Guests • Room #{{ $room->room_number }}
            </p>
        </div>
    </div>
</section>

<!-- Main Content -->
<section class="py-16 bg-white relative overflow-hidden">
    <div class="container mx-auto px-4">
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-12">
            <!-- Left Column - Images & Details -->
            <div class="lg:col-span-2">
                <!-- Main Image with Price Badge -->
                <div class="relative mb-12 group">
                    @if($room->image)
                        <div class="relative h-96 lg:h-[500px] rounded-2xl overflow-hidden shadow-2xl">
                            <img src="{{ asset('storage/' . $room->image) }}" 
                                 alt="{{ $room->name }}" 
                                 class="w-full h-full object-cover group-hover:scale-105 transition-transform duration-700">
                            <div class="absolute inset-0 bg-gradient-to-t from-black/50 to-transparent"></div>
                            
                            <!-- Price Badge -->
                            <div class="absolute top-6 right-6 px-6 py-3 bg-white/95 backdrop-blur-sm rounded-xl shadow-xl">
                                <div class="flex items-baseline gap-2">
                                    <span class="text-3xl font-bold text-[#558B2F]">LKR {{ number_format($room->price, 0) }}</span>
                                    <span class="text-gray-600 text-sm font-semibold">/ night</span>
                                </div>
                            </div>

                            <!-- Status Badge -->
                            @if($room->is_available)
                                <div class="absolute top-6 left-6 px-4 py-2 bg-[#558B2F]/90 backdrop-blur-sm rounded-full">
                                    <span class="text-white text-sm font-bold uppercase tracking-wider">Available</span>
                                </div>
                            @else
                                <div class="absolute top-6 left-6 px-4 py-2 bg-red-500/90 backdrop-blur-sm rounded-full">
                                    <span class="text-white text-sm font-bold uppercase tracking-wider">Booked</span>
                                </div>
                            @endif
                        </div>
                    @else
                        <div class="relative h-96 lg:h-[500px] rounded-2xl overflow-hidden shadow-2xl bg-gradient-to-br from-[#558B2F] to-[#3d6320] flex items-center justify-center">
                            <svg class="w-32 h-32 text-white/20" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="1.5" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"/>
                            </svg>
                        </div>
                    @endif
                </div>

                <!-- Room Quick Stats -->
                <div class="grid grid-cols-3 gap-6 mb-12">
                    <div class="group bg-gradient-to-br from-[#558B2F]/10 to-white rounded-2xl p-6 text-center hover:shadow-xl transition-all duration-300 hover:-translate-y-1 border border-[#558B2F]/20">
                        <div class="w-16 h-16 mx-auto mb-4 bg-[#558B2F] rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300 shadow-lg">
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z"/>
                            </svg>
                        </div>
                        <p class="text-sm text-gray-600 mb-1">Capacity</p>
                        <p class="text-xl font-bold text-gray-900">{{ $room->capacity }} Guests</p>
                    </div>

                    @if($room->view)
                    <div class="group bg-gradient-to-br from-amber-50 to-white rounded-2xl p-6 text-center hover:shadow-xl transition-all duration-300 hover:-translate-y-1 border border-amber-100">
                        <div class="w-16 h-16 mx-auto mb-4 bg-amber-500 rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300 shadow-lg">
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"/>
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"/>
                            </svg>
                        </div>
                        <p class="text-sm text-gray-600 mb-1">View</p>
                        <p class="text-xl font-bold text-gray-900">{{ $room->view }}</p>
                    </div>
                    @endif

                    <div class="group bg-gradient-to-br from-[#8bc34a]/10 to-white rounded-2xl p-6 text-center hover:shadow-xl transition-all duration-300 hover:-translate-y-1 border border-[#8bc34a]/20">
                        <div class="w-16 h-16 mx-auto mb-4 bg-[#8bc34a] rounded-xl flex items-center justify-center group-hover:scale-110 transition-transform duration-300 shadow-lg">
                            <svg class="w-8 h-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6"/>
                            </svg>
                        </div>
                        <p class="text-sm text-gray-600 mb-1">Type</p>
                        <p class="text-xl font-bold text-gray-900">{{ $room->type }}</p>
                    </div>
                </div>

                <!-- AI Slot Analyzer - Home Page Theme -->
                <div class="mb-12 group hidden">
                    <div class="relative rounded-2xl overflow-hidden shadow-2xl border border-[#558B2F]/20">
                        <!-- Gradient Background -->
                        <div class="absolute inset-0 bg-gradient-to-br from-[#3d6320] via-[#558B2F] to-[#4a7a28]"></div>
                        
                        <!-- Animated Pattern -->
                        <div class="absolute inset-0 opacity-10">
                            <div class="absolute inset-0" style="background-image: radial-gradient(circle at 2px 2px, rgba(255,255,255,0.15) 1px, transparent 0); background-size: 40px 40px;"></div>
                        </div>

                        <!-- Floating Orbs -->
                        <div class="absolute -top-24 -left-24 w-64 h-64 bg-[#8bc34a]/20 rounded-full blur-3xl animate-pulse"></div>
                        <div class="absolute -bottom-24 -right-24 w-64 h-64 bg-amber-400/20 rounded-full blur-3xl animate-pulse" style="animation-delay: 1.5s"></div>

                        <div class="relative p-8 md:p-10">
                            <div class="flex flex-col lg:flex-row gap-10">
                                <!-- AI Identity Section -->
                                <div class="w-full lg:w-2/5 border-b lg:border-b-0 lg:border-r border-white/10 pb-10 lg:pb-0 lg:pr-10">
                                    <div class="flex items-center gap-4 mb-8">
                                        <div class="relative h-16 w-16 group-hover:scale-110 transition-transform duration-500">
                                            <div class="absolute inset-0 bg-[#8bc34a] rounded-2xl blur-lg opacity-30 group-hover:opacity-50 transition-opacity"></div>
                                            <div class="relative h-full w-full bg-gradient-to-br from-[#558B2F] to-[#457025] rounded-2xl flex items-center justify-center shadow-xl">
                                                <svg class="h-8 w-8 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 10V3L4 14h7v7l9-11h-7z"/>
                                                </svg>
                                            </div>
                                        </div>
                                        <div>
                                            <span class="text-[10px] font-black text-[#8bc34a] uppercase tracking-[0.3em] mb-1 block opacity-80">Smart Booking</span>
                                            <h3 class="text-3xl font-display font-bold text-white leading-tight">
                                                AI Availability<br>
                                                <span class="bg-gradient-to-r from-[#8bc34a] via-[#c6d95f] to-amber-400 bg-clip-text text-transparent">Analyzer</span>
                                            </h3>
                                        </div>
                                    </div>
                                    
                                    <div class="space-y-6">
                                        <p class="text-white/70 text-sm leading-relaxed" id="ai-status">Analyzing booking patterns and availability...</p>
                                        
                                        <div id="ai-loader" class="flex items-center gap-4 py-2">
                                            <div class="flex gap-1.5">
                                                <div class="w-2 h-2 bg-[#8bc34a] rounded-full animate-bounce" style="animation-delay: -0.3s"></div>
                                                <div class="w-2 h-2 bg-[#8bc34a] rounded-full animate-bounce" style="animation-delay: -0.15s"></div>
                                                <div class="w-2 h-2 bg-[#8bc34a] rounded-full animate-bounce"></div>
                                            </div>
                                            <span class="text-[10px] font-bold text-[#8bc34a]/60 uppercase tracking-widest">Processing</span>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- AI Analysis Insights -->
                                <div class="w-full lg:w-3/5">
                                    <div id="ai-result" class="hidden space-y-4 animate-fade-in-up">
                                        <!-- Insights will be injected here -->
                                    </div>

                                    <!-- Available Slots Display -->
                                    <div id="ai-slots-container" class="hidden mt-8">
                                        <div class="flex items-center justify-between mb-6">
                                            <h4 class="text-xs font-bold text-[#8bc34a]/80 uppercase tracking-[0.2em]">Recommended Dates</h4>
                                            <div class="h-px flex-1 bg-white/10 mx-4"></div>
                                        </div>
                                        <div class="grid grid-cols-1 sm:grid-cols-2 gap-4" id="ai-slots-list">
                                            <!-- Slots will be injected here -->
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div> 

                <!-- Description -->
                <div class="mb-12">
                    <h2 class="text-3xl font-display font-bold text-gray-900 mb-6 flex items-center">
                        <span class="w-2 h-8 bg-[#558B2F] rounded-full mr-4"></span>
                        About This Villa
                    </h2>
                    <p class="text-gray-600 leading-relaxed text-lg">
                        {{ $room->description ?? 'Experience the perfect blend of luxury and sustainability in this beautifully appointed villa. Designed with your comfort and the environment in mind, every detail has been carefully curated to ensure an unforgettable eco-friendly stay.' }}
                    </p>
                </div>

                <!-- Amenities & Facilities -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-8 mb-12">
                    <!-- Amenities -->
                    @if($room->amenities && $room->amenities->count() > 0)
                    <div class="bg-gradient-to-br from-[#558B2F]/10 to-white rounded-2xl p-8 border border-[#558B2F]/20 shadow-lg hover:shadow-xl transition-all duration-300">
                        <h2 class="text-2xl font-display font-bold text-gray-900 mb-6 flex items-center">
                            <span class="w-12 h-12 bg-[#558B2F] rounded-xl flex items-center justify-center mr-3 shadow-lg">
                                <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z"/>
                                </svg>
                            </span>
                            Villa Amenities
                        </h2>
                        <div class="space-y-4">
                            @foreach($room->amenities as $amenity)
                                <div class="flex items-center text-gray-700 group/item">
                                    <div class="w-12 h-12 bg-[#558B2F]/10 rounded-xl flex items-center justify-center mr-4 group-hover/item:bg-[#558B2F]/20 group-hover/item:scale-110 transition-all duration-300">
                                        @if($amenity->icon)
                                            <img src="{{ asset('storage/' . $amenity->icon) }}" class="w-6 h-6 object-contain">
                                        @else
                                            <svg class="w-6 h-6 text-[#558B2F]" fill="currentColor" viewBox="0 0 20 20">
                                                <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"/>
                                            </svg>
                                        @endif
                                    </div>
                                    <span class="font-semibold">{{ $amenity->name }}</span>
                                </div>
                            @endforeach
                        </div>
                    </div>
                    @endif

                    <!-- Facilities -->
                    @if($room->facilities && $room->facilities->count() > 0)
                    <div class="bg-gradient-to-br from-amber-50 to-white rounded-2xl p-8 border border-amber-100 shadow-lg hover:shadow-xl transition-all duration-300">
                        <h2 class="text-2xl font-display font-bold text-gray-900 mb-6 flex items-center">
                            <span class="w-12 h-12 bg-amber-500 rounded-xl flex items-center justify-center mr-3 shadow-lg">
                                <svg class="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4"/>
                                </svg>
                            </span>
                            Eco-Resort Facilities
                        </h2>
                        <div class="space-y-4">
                            @foreach($room->facilities as $facility)
                                <div class="flex items-center text-gray-700 group/item">
                                    <div class="w-12 h-12 bg-amber-100 rounded-xl flex items-center justify-center mr-4 group-hover/item:bg-amber-200 group-hover/item:scale-110 transition-all duration-300">
                                        @if($facility->icon)
                                            <img src="{{ asset('storage/' . $facility->icon) }}" class="w-6 h-6 object-contain">
                                        @else
                                            <svg class="w-6 h-6 text-amber-600" fill="currentColor" viewBox="0 0 20 20">
                                                <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"/>
                                            </svg>
                                        @endif
                                    </div>
                                    <span class="font-semibold">{{ $facility->name }}</span>
                                </div>
                            @endforeach
                        </div>
                    </div>
                    @endif
                </div>

                <!-- Photo Gallery -->
                @if($room->images && $room->images->count() > 0)
                <div class="mb-12">
                    <h2 class="text-3xl font-display font-bold text-gray-900 mb-6 flex items-center">
                        <span class="w-2 h-8 bg-[#558B2F] rounded-full mr-4"></span>
                        Photo Gallery
                    </h2>
                    <div class="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                        @foreach($room->images as $image)
                            <div class="relative h-48 rounded-xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 cursor-pointer group" onclick="openLightbox('{{ asset('storage/' . $image->image_path) }}')">
                                <img src="{{ asset('storage/' . $image->image_path) }}" 
                                     alt="Gallery Image" 
                                     class="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700">
                                <div class="absolute inset-0 bg-gradient-to-t from-black/50 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                                
                                <!-- Zoom Icon -->
                                <div class="absolute inset-0 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                                    <div class="w-12 h-12 bg-white/90 backdrop-blur-sm rounded-full flex items-center justify-center">
                                        <svg class="w-6 h-6 text-[#558B2F]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0zM10 7v6m3-3H7"/>
                                        </svg>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
                @endif
            </div>

            <!-- Right Column - Booking Form -->
            <div class="lg:col-span-1">
                <div class="sticky top-24">
                    <div class="bg-gradient-to-br from-[#3d6320] via-[#558B2F] to-[#4a7a28] rounded-2xl p-8 shadow-2xl border border-[#8bc34a]/20">
                        <h3 class="text-2xl font-display font-bold text-white mb-6">Book Your Stay</h3>
                        
                        @auth
                            <!-- Availability Message -->
                            <div id="availability-message" class="hidden mb-6 p-4 rounded-xl transition-all duration-300"></div>

                            <form action="{{ route('bookings.store') }}" method="POST" id="booking-form" class="space-y-6">
                                @csrf
                                <input type="hidden" name="room_id" value="{{ $room->id }}">
                                
                                <!-- Check-in Date -->
                                <div class="form-group">
                                    <label for="check_in" class="block text-sm font-bold text-[#8bc34a] mb-3 uppercase tracking-wider flex items-center gap-2">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                        </svg>
                                        Check-in Date
                                    </label>
                                    <input type="date" 
                                           name="check_in" 
                                           id="check_in" 
                                           min="{{ date('Y-m-d', strtotime('+1 day')) }}" 
                                           class="w-full px-4 py-4 bg-white/10 border-2 border-white/20 rounded-xl text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-[#8bc34a] focus:border-[#8bc34a] transition-all duration-300" 
                                           required>
                                    @error('check_in')
                                        <p class="text-red-400 text-sm mt-2">{{ $message }}</p>
                                    @enderror
                                </div>

                                <!-- Check-out Date -->
                                <div class="form-group">
                                    <label for="check_out" class="block text-sm font-bold text-[#8bc34a] mb-3 uppercase tracking-wider flex items-center gap-2">
                                        <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z"/>
                                        </svg>
                                        Check-out Date
                                    </label>
                                    <input type="date" 
                                           name="check_out" 
                                           id="check_out" 
                                           min="{{ date('Y-m-d', strtotime('+2 days')) }}" 
                                           class="w-full px-4 py-4 bg-white/10 border-2 border-white/20 rounded-xl text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-[#8bc34a] focus:border-[#8bc34a] transition-all duration-300" 
                                           required>
                                    @error('check_out')
                                        <p class="text-red-400 text-sm mt-2">{{ $message }}</p>
                                    @enderror
                                </div>

                                <!-- Price Summary -->
                                <div class="bg-white/5 rounded-xl p-6 border border-white/10" id="price-summary">
                                    <div class="flex justify-between items-center text-white mb-3">
                                        <span class="text-sm">Price per night</span>
                                        <span class="font-bold text-lg">LKR {{ number_format($room->price, 0) }}</span>
                                    </div>
                                    <div class="flex justify-between items-center text-[#8bc34a] text-sm mb-3">
                                        <span>Total nights</span>
                                        <span id="total-nights" class="font-semibold">-</span>
                                    </div>
                                    <div class="border-t border-white/10 pt-4 mt-4 hidden" id="total-price-container">
                                        <div class="flex justify-between items-center">
                                            <span class="text-white font-bold text-xl">Total Price</span>
                                            <span id="total-price" class="bg-gradient-to-r from-[#8bc34a] via-[#c6d95f] to-amber-400 bg-clip-text text-transparent font-bold text-2xl">LKR 0</span>
                                        </div>
                                    </div>
                                </div>

                                <!-- Submit Button -->
                                <button type="submit" 
                                        id="book-now-btn"
                                        class="w-full bg-amber-400 hover:bg-amber-500 text-gray-900 py-5 rounded-md font-bold text-lg transition-all duration-300 transform hover:scale-105 shadow-xl hover:shadow-2xl relative overflow-hidden group uppercase tracking-widest">
                                    <span class="relative z-10 flex items-center justify-center" id="btn-text">
                                        <svg class="w-6 h-6 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/>
                                        </svg>
                                        Book Now
                                    </span>
                                    <span class="hidden absolute inset-0 flex items-center justify-center" id="btn-loading">
                                        <svg class="animate-spin h-6 w-6 text-gray-900" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                            <circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle>
                                            <path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                        </svg>
                                    </span>
                                </button>

                                <p class="text-xs text-white/60 text-center flex items-center justify-center gap-2">
                                    <svg class="w-4 h-4 text-[#8bc34a]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                                    </svg>
                                    Free cancellation up to 24 hours before check-in
                                </p>
                            </form>
                        @else
                            <div class="text-center py-12">
                                <div class="w-20 h-20 mx-auto mb-6 bg-[#8bc34a]/20 rounded-full flex items-center justify-center">
                                    <svg class="w-10 h-10 text-[#8bc34a]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"/>
                                    </svg>
                                </div>
                                <p class="text-white text-lg mb-8">Please login to book this villa</p>
                                <a href="{{ route('login') }}" 
                                   class="inline-block w-full bg-amber-400 hover:bg-amber-500 text-gray-900 py-5 rounded-md font-bold text-lg transition-all transform hover:scale-105 shadow-xl uppercase tracking-widest">
                                    Login to Book
                                </a>
                            </div>
                        @endauth
                    </div>

                    <!-- Contact Info -->
                    <div class="mt-6 bg-gradient-to-br from-gray-50 to-[#558B2F]/5 rounded-xl p-6 border border-[#558B2F]/10 shadow-lg">
                        <h4 class="font-bold text-gray-900 mb-3 flex items-center">
                            <svg class="w-5 h-5 text-[#558B2F] mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M18.364 5.636l-3.536 3.536m0 5.656l3.536 3.536M9.172 9.172L5.636 5.636m3.536 9.192l-3.536 3.536M21 12a9 9 0 11-18 0 9 9 0 0118 0zm-5 0a4 4 0 11-8 0 4 4 0 018 0z"/>
                            </svg>
                            Need Assistance?
                        </h4>
                        <p class="text-sm text-gray-600 mb-4">Our eco-concierge team is ready to help</p>
                        <a href="{{ route('contact') }}" class="inline-flex items-center gap-2 text-[#558B2F] hover:text-[#457025] font-semibold text-sm transition-all group">
                            Contact Us
                            <svg class="w-4 h-4 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/>
                            </svg>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Similar Rooms -->
<section class="py-24 bg-gradient-to-br from-gray-50 to-[#558B2F]/5 relative overflow-hidden">
    <div class="absolute top-0 right-0 w-96 h-96 bg-[#558B2F]/10 rounded-full blur-3xl"></div>
    
    <div class="container mx-auto px-4 relative z-10">
        <div class="text-center mb-12">
            <span class="inline-block px-6 py-2 bg-[#558B2F]/10 text-[#558B2F] rounded-full text-sm font-bold uppercase tracking-wider mb-4">
                More Options
            </span>
            <h2 class="text-4xl md:text-5xl font-display font-bold text-gray-900 mb-4">
                Discover More Villas
            </h2>
            <p class="text-gray-600 text-lg mb-8">Explore our collection of eco-luxury accommodations</p>
            <a href="{{ route('rooms.index') }}" class="inline-flex items-center gap-3 px-8 py-4 bg-[#558B2F] hover:bg-[#457025] text-white font-bold rounded-md transition-all duration-300 shadow-lg hover:shadow-2xl hover:scale-105 uppercase tracking-widest text-sm">
                View All Villas
                <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 5l7 7-7 7"/>
                </svg>
            </a>
        </div>
    </div>
</section>

@auth
<script>
// Enhanced booking form with animations and real-time validation
document.addEventListener('DOMContentLoaded', function() {
    const checkIn = document.getElementById('check_in');
    const checkOut = document.getElementById('check_out');
    const totalNights = document.getElementById('total-nights');
    const totalPrice = document.getElementById('total-price');
    const totalPriceContainer = document.getElementById('total-price-container');
    const availabilityMessage = document.getElementById('availability-message');
    const bookingForm = document.getElementById('booking-form');
    const bookNowBtn = document.getElementById('book-now-btn');
    const btnText = document.getElementById('btn-text');
    const btnLoading = document.getElementById('btn-loading');
    const roomId = {{ $room->id }};
    const pricePerNight = {{ $room->price }};
    
    let checkTimeout;

    // Calculate total nights and price
    function calculateNights() {
        if (checkIn.value && checkOut.value) {
            const start = new Date(checkIn.value);
            const end = new Date(checkOut.value);
            const nights = Math.ceil((end - start) / (1000 * 60 * 60 * 24));
            
            if (nights > 0) {
                totalNights.textContent = nights + ' night' + (nights > 1 ? 's' : '');
                const total = nights * pricePerNight;
                totalPrice.textContent = 'LKR ' + total.toLocaleString();
                totalPriceContainer.classList.remove('hidden');
                totalPriceContainer.classList.add('animate-fade-in-up');
                
                // Check availability after dates are selected
                checkAvailability();
            } else {
                totalNights.textContent = '-';
                totalPriceContainer.classList.add('hidden');
            }
        }
    }

    // Real-time availability check via AJAX
    function checkAvailability() {
        if (!checkIn.value || !checkOut.value) return;

        clearTimeout(checkTimeout);
        checkTimeout = setTimeout(() => {
            fetch('{{ route("bookings.check-availability") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({
                    room_id: roomId,
                    check_in: checkIn.value,
                    check_out: checkOut.value
                })
            })
            .then(response => response.json())
            .then(data => {
                showAvailabilityMessage(data.available, data.message);
                bookNowBtn.disabled = !data.available;
                
                if (!data.available) {
                    bookNowBtn.classList.add('opacity-50', 'cursor-not-allowed');
                } else {
                    bookNowBtn.classList.remove('opacity-50', 'cursor-not-allowed');
                }
            })
            .catch(error => {
                console.error('Error checking availability:', error);
            });
        }, 500);
    }

    // Show availability message with animation
    function showAvailabilityMessage(isAvailable, message) {
        availabilityMessage.classList.remove('hidden', 'bg-[#558B2F]/20', 'bg-red-500/20', 'text-[#8bc34a]', 'text-red-300');
        
        if (isAvailable) {
            availabilityMessage.classList.add('bg-[#558B2F]/20', 'text-[#8bc34a]', 'border-2', 'border-[#558B2F]/30');
            availabilityMessage.innerHTML = `
                <div class="flex items-center">
                    <svg class="w-5 h-5 mr-3 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"/>
                    </svg>
                    <span class="font-semibold">${message}</span>
                </div>
            `;
        } else {
            availabilityMessage.classList.add('bg-red-500/20', 'text-red-300', 'border-2', 'border-red-500/30');
            availabilityMessage.innerHTML = `
                <div class="flex items-center">
                    <svg class="w-5 h-5 mr-3 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                    </svg>
                    <span class="font-semibold">${message}</span>
                </div>
            `;
        }
    }

    // Form submission with loading state
    bookingForm.addEventListener('submit', function(e) {
        btnText.classList.add('hidden');
        btnLoading.classList.remove('hidden');
        bookNowBtn.disabled = true;
        bookNowBtn.classList.add('opacity-75');
    });

    // Event listeners
    if (checkIn && checkOut) {
        checkIn.addEventListener('change', calculateNights);
        checkOut.addEventListener('change', calculateNights);
        
        // Update min date for checkout when checkin changes
        checkIn.addEventListener('change', function() {
            const minCheckout = new Date(this.value);
            minCheckout.setDate(minCheckout.getDate() + 1);
            checkOut.min = minCheckout.toISOString().split('T')[0];
        });
    }
});

// AI Analysis Logic
document.addEventListener('DOMContentLoaded', function() {
    const aiStatus = document.getElementById('ai-status');
    const aiLoader = document.getElementById('ai-loader');
    const aiResult = document.getElementById('ai-result');
    const aiSlotsContainer = document.getElementById('ai-slots-container');
    const aiSlotsList = document.getElementById('ai-slots-list');
    const roomId = {{ $room->id }};

    function runAIAnalysis() {
        fetch(`/rooms/${roomId}/ai-analysis`)
            .then(response => response.json())
            .then(data => {
                setTimeout(() => {
                    displayAIResults(data);
                }, 2000);
            })
            .catch(error => {
                aiStatus.textContent = "AI analyzer is currently offline. Please select dates manually.";
                aiLoader.classList.add('hidden');
            });
    }

    function displayAIResults(data) {
        aiLoader.classList.add('hidden');
        aiStatus.textContent = data.summary;
        aiStatus.classList.remove('text-white/70');
        aiStatus.classList.add('text-[#8bc34a]', 'font-semibold');
        
        // Show Insights
        aiResult.innerHTML = '';
        data.insights.forEach((insight, index) => {
            const div = document.createElement('div');
            div.className = 'flex items-start gap-4 bg-white/5 p-5 rounded-xl border border-[#558B2F]/10 hover:bg-[#558B2F]/5 hover:border-[#558B2F]/30 transition-all cursor-default group/ins';
            div.style.animation = `fadeIn 0.5s ease-out ${index * 0.15}s forwards`;
            div.style.opacity = '0';
            div.innerHTML = `
                <div class="h-10 w-10 rounded-xl bg-[#558B2F]/10 flex items-center justify-center flex-shrink-0 group-hover/ins:scale-110 group-hover/ins:bg-[#558B2F]/20 transition-all">
                    <svg class="h-5 w-5 text-[#8bc34a]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                    </svg>
                </div>
                <span class="text-sm text-white/80 font-medium leading-relaxed">${insight}</span>
            `;
            aiResult.appendChild(div);
        });
        aiResult.classList.remove('hidden');

        // Show Slots
        if (data.slots.length > 0) {
            aiSlotsList.innerHTML = '';
            data.slots.forEach((slot, index) => {
                const btn = document.createElement('button');
                btn.className = 'rounded-xl p-6 text-left transition-all relative overflow-hidden group/card bg-white/5 border border-[#558B2F]/10 hover:bg-[#558B2F]/10 hover:border-[#558B2F]/30 hover:shadow-lg';
                btn.style.animation = `fadeIn 0.5s ease-out ${(index + 2) * 0.15}s forwards`;
                btn.style.opacity = '0';
                btn.innerHTML = `
                    <div class="absolute top-0 right-0 p-4 opacity-0 group-hover/card:opacity-100 transition-opacity">
                        <svg class="w-5 h-5 text-[#8bc34a]" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M14 5l7 7m0 0l-7 7m7-7H3"/>
                        </svg>
                    </div>
                    <div class="mb-3">
                        <span class="px-3 py-1.5 rounded-full bg-[#558B2F]/20 text-[10px] font-black text-[#8bc34a] uppercase tracking-widest border border-[#558B2F]/30">Recommended</span>
                    </div>
                    <p class="text-white text-lg font-bold mb-2 font-display">${formatDate(slot.start)} — ${formatDate(slot.end)}</p>
                    <p class="text-white/60 text-sm">${slot.days} Night${slot.days > 1 ? 's' : ''} • High Availability</p>
                `;
                btn.onclick = () => {
                    document.getElementById('check_in').value = slot.start;
                    document.getElementById('check_out').value = slot.end;
                    document.getElementById('check_in').dispatchEvent(new Event('change'));
                    document.getElementById('booking-form').scrollIntoView({ behavior: 'smooth', block: 'center' });
                };
                aiSlotsList.appendChild(btn);
            });
            aiSlotsContainer.classList.remove('hidden');
        }
    }

    function formatDate(dateStr) {
        const date = new Date(dateStr);
        return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
    }

    runAIAnalysis();
});
</script>
@endauth

<script>
// Lightbox functionality for gallery
window.openLightbox = function(imageSrc) {
    const lightbox = document.createElement('div');
    lightbox.id = 'lightbox';
    lightbox.className = 'fixed inset-0 z-[100] bg-black/95 backdrop-blur-sm flex items-center justify-center p-4 cursor-pointer animate-fade-in-up';
    lightbox.innerHTML = `
        <div class="relative max-w-6xl w-full h-full flex items-center justify-center">
            <img src="${imageSrc}" class="max-w-full max-h-full object-contain rounded-2xl shadow-2xl" style="animation: scaleUp 0.3s ease-out forwards;">
            <button class="absolute top-8 right-8 w-14 h-14 bg-white/10 backdrop-blur-md hover:bg-white/20 border border-white/20 rounded-full flex items-center justify-center text-white transition-all hover:scale-110" onclick="this.closest('#lightbox').remove()">
                <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                </svg>
            </button>
        </div>
    `;
    lightbox.onclick = (e) => {
        if (e.target.id === 'lightbox') lightbox.remove();
    };
    document.body.appendChild(lightbox);
    document.body.style.overflow = 'hidden';
    
    const observer = new MutationObserver((mutations) => {
        if (!document.getElementById('lightbox')) {
            document.body.style.overflow = 'auto';
            observer.disconnect();
        }
    });
    observer.observe(document.body, { childList: true });
};
</script>

<!-- Custom CSS for Animations -->
<style>
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

@keyframes scaleUp {
    from { transform: scale(0.95); opacity: 0; }
    to { transform: scale(1); opacity: 1; }
}

@keyframes particle-float {
    0%, 100% {
        transform: translateY(0) translateX(0);
        opacity: 0.3;
    }
    50% {
        transform: translateY(-100px) translateX(50px);
        opacity: 0.6;
    }
}

.particle-float {
    animation: particle-float 8s ease-in-out infinite;
}

.animate-fade-in-up {
    animation: fadeIn 0.8s ease-out forwards;
    opacity: 0;
}

html {
    scroll-behavior: smooth;
}
</style>

@endsection