import './bootstrap';

import Alpine from 'alpinejs';
import Swiper from 'swiper/bundle';
import 'swiper/css/bundle';
import AOS from 'aos';
import 'aos/dist/aos.css';

window.Alpine = Alpine;
window.Swiper = Swiper;

Alpine.start();

// ============================================
// ENHANCED ANIMATIONS INITIALIZATION
// ============================================

document.addEventListener('DOMContentLoaded', () => {
    // Initialize AOS with enhanced settings
    AOS.init({
        duration: 1000,
        once: true,
        offset: 100,
        easing: 'ease-out-cubic',
        delay: 50,
        anchorPlacement: 'top-bottom',
        disable: window.innerWidth < 768 ? 'mobile' : false,
    });

    // Page Load Animation
    document.body.classList.add('animate-fade-in');

    // Smooth Scroll for Anchor Links
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            const href = this.getAttribute('href');
            if (href !== '#' && href !== '') {
                e.preventDefault();
                const target = document.querySelector(href);
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            }
        });
    });

    // Parallax Effect for Hero Sections
    const parallaxElements = document.querySelectorAll('[data-parallax]');
    if (parallaxElements.length > 0) {
        window.addEventListener('scroll', () => {
            const scrolled = window.pageYOffset;
            parallaxElements.forEach(element => {
                const speed = element.dataset.parallax || 0.5;
                element.style.transform = `translateY(${scrolled * speed}px)`;
            });
        });
    }

    // Intersection Observer for Lazy Animations
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate-fade-in-up');
                observer.unobserve(entry.target);
            }
        });
    }, observerOptions);

    // Observe elements with lazy-animate class
    document.querySelectorAll('.lazy-animate').forEach(element => {
        observer.observe(element);
    });

    // Enhanced Hover Effects for Images
    const imageContainers = document.querySelectorAll('.image-zoom-container');
    imageContainers.forEach(container => {
        const image = container.querySelector('img');
        if (image) {
            image.classList.add('image-zoom');
        }
    });

    // Form Input Animations
    const inputs = document.querySelectorAll('input, textarea, select');
    inputs.forEach(input => {
        input.addEventListener('focus', function () {
            this.parentElement?.classList.add('input-focused');
        });

        input.addEventListener('blur', function () {
            this.parentElement?.classList.remove('input-focused');
        });
    });

    // Loading State for Forms
    const forms = document.querySelectorAll('form[data-loading]');
    forms.forEach(form => {
        form.addEventListener('submit', function () {
            const submitBtn = this.querySelector('button[type="submit"]');
            if (submitBtn) {
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<span class="spinner mr-2"></span> Processing...';
            }
        });
    });

    // Stagger Animation for Grid Items
    const gridContainers = document.querySelectorAll('[data-stagger]');
    gridContainers.forEach(container => {
        const items = container.children;
        Array.from(items).forEach((item, index) => {
            item.style.animationDelay = `${index * 0.1}s`;
            item.classList.add('stagger-item');
        });
    });

    // Number Counter Animation
    const counters = document.querySelectorAll('[data-counter]');
    const counterObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const counter = entry.target;
                const target = parseInt(counter.dataset.counter);
                const duration = 2000;
                const increment = target / (duration / 16);
                let current = 0;

                const updateCounter = () => {
                    current += increment;
                    if (current < target) {
                        counter.textContent = Math.ceil(current);
                        requestAnimationFrame(updateCounter);
                    } else {
                        counter.textContent = target;
                    }
                };

                updateCounter();
                counterObserver.unobserve(counter);
            }
        });
    }, { threshold: 0.5 });

    counters.forEach(counter => counterObserver.observe(counter));

    // Refresh AOS on dynamic content
    window.addEventListener('load', () => {
        AOS.refresh();
    });
});

// ============================================
// TOAST NOTIFICATION SYSTEM
// ============================================

class ToastNotification {
    constructor() {
        this.container = document.createElement('div');
        this.container.id = 'toast-container';
        this.container.className = 'fixed top-4 right-4 z-50 flex flex-col gap-3';
        document.body.appendChild(this.container);
    }

    show(message, type = 'success', duration = 5000) {
        const toast = document.createElement('div');

        // Icon based on type
        let icon = '';
        let bgClass = '';
        let textClass = '';
        let borderClass = '';

        if (type === 'success') {
            bgClass = 'bg-white/95 backdrop-blur shadow-lg shadow-green-900/10';
            borderClass = 'border-l-4 border-green-500';
            textClass = 'text-gray-800';
            icon = `<svg class="w-6 h-6 text-green-500 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>`;
        } else if (type === 'error') {
            bgClass = 'bg-white/95 backdrop-blur shadow-lg shadow-red-900/10';
            borderClass = 'border-l-4 border-red-500';
            textClass = 'text-gray-800';
            icon = `<svg class="w-6 h-6 text-red-500 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>`;
        } else {
            bgClass = 'bg-white/95 backdrop-blur shadow-lg shadow-blue-900/10';
            borderClass = 'border-l-4 border-blue-500';
            textClass = 'text-gray-800';
            icon = `<svg class="w-6 h-6 text-blue-500 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>`;
        }

        toast.className = `${bgClass} ${borderClass} ${textClass} px-6 py-4 rounded-r shadow-2xl flex items-center transform transition-all duration-500 translate-x-full opacity-0 min-w-[300px] max-w-md`;
        toast.innerHTML = `
            ${icon}
            <div>
                <p class="font-medium">${message}</p>
            </div>
            <button onclick="this.parentElement.remove()" class="ml-auto text-gray-400 hover:text-gray-600">
                <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg>
            </button>
        `;

        this.container.appendChild(toast);

        // Animate in
        requestAnimationFrame(() => {
            toast.classList.remove('translate-x-full', 'opacity-0');
            toast.classList.add('translate-x-0', 'opacity-100');
        });

        // Auto remove
        setTimeout(() => {
            toast.classList.remove('translate-x-0', 'opacity-100');
            toast.classList.add('translate-x-full', 'opacity-0');
            setTimeout(() => {
                toast.remove();
            }, 500);
        }, duration);
    }
}

// Expose to window
window.Toast = new ToastNotification();
