<?php

namespace App\Services;

use App\Models\Room;
use App\Models\Booking;
use Carbon\Carbon;
use Carbon\CarbonPeriod;

class RoomAnalysisService
{
    /**
     * Analyze room availability and return smart suggestions.
     */
    public function analyze(Room $room)
    {
        $today = Carbon::today();
        $endPeriod = $today->copy()->addDays(60);
        
        // Get all booked dates in the next 60 days
        $bookings = Booking::where('room_id', $room->id)
            ->whereIn('status', ['confirmed', 'pending'])
            ->where(function($query) use ($today, $endPeriod) {
                $query->where('check_in', '<=', $endPeriod)
                      ->where('check_out', '>=', $today);
            })
            ->orderBy('check_in')
            ->get();

        $bookedDates = [];
        foreach ($bookings as $booking) {
            $period = CarbonPeriod::create($booking->check_in, $booking->check_out->copy()->subDay());
            foreach ($period as $date) {
                $bookedDates[] = $date->format('Y-m-d');
            }
        }

        $availableSlots = $this->findAvailableSlots($today, $endPeriod, $bookedDates);
        $insights = $this->generateInsights($room, $availableSlots, $bookings);

        return [
            'room_id' => $room->id,
            'room_name' => $room->name,
            'slots' => array_slice($availableSlots, 0, 3), // Top 3 best slots
            'insights' => $insights,
            'summary' => $this->generateSummary($availableSlots, $insights)
        ];
    }

    private function findAvailableSlots($start, $end, $bookedDates)
    {
        $period = CarbonPeriod::create($start, $end);
        $slots = [];
        $currentSlot = null;

        foreach ($period as $date) {
            $dateStr = $date->format('Y-m-d');
            
            if (!in_array($dateStr, $bookedDates)) {
                if (!$currentSlot) {
                    $currentSlot = [
                        'start' => $dateStr,
                        'end' => $dateStr,
                        'days' => 1
                    ];
                } else {
                    $currentSlot['end'] = $dateStr;
                    $currentSlot['days']++;
                }
            } else {
                if ($currentSlot) {
                    $slots[] = $currentSlot;
                    $currentSlot = null;
                }
            }
        }

        if ($currentSlot) {
            $slots[] = $currentSlot;
        }

        // Sort by length of stay
        usort($slots, function($a, $b) {
            return $b['days'] <=> $a['days'];
        });

        return $slots;
    }

    private function generateInsights($room, $slots, $bookings)
    {
        $insights = [];
        
        $totalBookings = $bookings->count();
        if ($totalBookings > 5) {
            $insights[] = "This room is in high demand! It has seen {$totalBookings} bookings in the last 60 days.";
        }

        $weekendBookings = 0;
        foreach ($bookings as $booking) {
            if ($booking->check_in->isWeekend() || $booking->check_out->isWeekend()) {
                $weekendBookings++;
            }
        }

        if ($weekendBookings > ($totalBookings / 2)) {
            $insights[] = "Weekends are usually busy here. Consider a mid-week stay for a more peaceful experience.";
        }

        if (count($slots) > 0 && $slots[0]['days'] > 7) {
            $insights[] = "There's a great opportunity for a long stay starting on " . Carbon::parse($slots[0]['start'])->format('M d') . ".";
        }

        return $insights;
    }

    private function generateSummary($slots, $insights)
    {
        if (count($slots) == 0) {
            return "This room is fully booked for the next 60 days. Check back soon for cancellations!";
        }

        $bestSlot = $slots[0];
        return "We found " . count($slots) . " available slots. The longest available stay is " . $bestSlot['days'] . " days starting from " . Carbon::parse($bestSlot['start'])->format('M d') . ".";
    }
}
