<?php

namespace App\Services;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use App\Models\Booking;
use App\Models\Room;

class ChannelManagerService
{
    protected $client;
    protected $propertyId;
    protected $apiKey;
    protected $baseUrl;

    public function __construct()
    {
        $this->client = new Client([
            'timeout' => 30,
            'verify' => false, // Set to true in production with proper SSL
        ]);
        
        $this->propertyId = config('services.channel_manager.property_id');
        $this->apiKey = config('services.channel_manager.api_key');
        $this->baseUrl = config('services.channel_manager.base_url', 'https://api.cloudbeds.com/api/v1.1/');
    }

    /**
     * Get all reservations from channel manager
     */
    public function getReservations($startDate = null, $endDate = null)
    {
        try {
            $startDate = $startDate ?? now()->format('Y-m-d');
            $endDate = $endDate ?? now()->addMonths(3)->format('Y-m-d');

            $response = $this->client->get($this->baseUrl . 'getReservations', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->apiKey,
                    'Accept' => 'application/json',
                ],
                'query' => [
                    'propertyID' => $this->propertyId,
                    'startDate' => $startDate,
                    'endDate' => $endDate,
                ]
            ]);

            $data = json_decode($response->getBody(), true);
            
            // Sync reservations to local database
            if (isset($data['data'])) {
                foreach ($data['data'] as $reservation) {
                    $this->syncReservation($reservation);
                }
            }

            return $data;
        } catch (\Exception $e) {
            Log::error('Channel Manager - Get Reservations Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Sync a single reservation to local database
     */
    protected function syncReservation($reservation)
    {
        try {
            // Map external room ID to local room
            $room = $this->mapExternalRoom($reservation['roomTypeID'] ?? null);
            
            if (!$room) {
                Log::warning('Channel Manager - Room mapping not found for: ' . ($reservation['roomTypeID'] ?? 'unknown'));
                return null;
            }

            return Booking::updateOrCreate(
                [
                    'external_id' => $reservation['reservationID'],
                    'source' => $reservation['source'] ?? 'channel_manager',
                ],
                [
                    'room_id' => $room->id,
                    'user_id' => null, // External bookings don't have local users
                    'check_in' => $reservation['startDate'],
                    'check_out' => $reservation['endDate'],
                    'guests' => $reservation['adults'] ?? 2,
                    'total_price' => $reservation['balance'] ?? 0,
                    'status' => $this->mapStatus($reservation['status'] ?? 'confirmed'),
                    'guest_name' => $reservation['guestName'] ?? 'Guest',
                    'guest_email' => $reservation['guestEmail'] ?? null,
                    'guest_phone' => $reservation['guestPhone'] ?? null,
                    'external_data' => json_encode($reservation),
                ]
            );
        } catch (\Exception $e) {
            Log::error('Channel Manager - Sync Reservation Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Update room availability across all channels
     */
    public function updateAvailability($roomId, $date, $quantity)
    {
        try {
            // Get external room mapping
            $externalRoomId = $this->getExternalRoomId($roomId);
            
            if (!$externalRoomId) {
                Log::warning('Channel Manager - No external mapping for room: ' . $roomId);
                return false;
            }

            $response = $this->client->post($this->baseUrl . 'putRoomAvailability', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'propertyID' => $this->propertyId,
                    'roomTypeID' => $externalRoomId,
                    'date' => $date,
                    'quantity' => $quantity,
                ]
            ]);

            return $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            Log::error('Channel Manager - Update Availability Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Update room rates across all channels
     */
    public function updateRates($roomId, $date, $price)
    {
        try {
            $externalRoomId = $this->getExternalRoomId($roomId);
            
            if (!$externalRoomId) {
                return false;
            }

            $response = $this->client->post($this->baseUrl . 'putRoomRates', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'propertyID' => $this->propertyId,
                    'roomTypeID' => $externalRoomId,
                    'date' => $date,
                    'rate' => $price,
                ]
            ]);

            return $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            Log::error('Channel Manager - Update Rates Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Block dates (make unavailable)
     */
    public function blockDates($roomId, $startDate, $endDate)
    {
        try {
            $externalRoomId = $this->getExternalRoomId($roomId);
            
            if (!$externalRoomId) {
                return false;
            }

            $response = $this->client->post($this->baseUrl . 'putRoomBlock', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->apiKey,
                    'Content-Type' => 'application/json',
                ],
                'json' => [
                    'propertyID' => $this->propertyId,
                    'roomTypeID' => $externalRoomId,
                    'startDate' => $startDate,
                    'endDate' => $endDate,
                    'isBlocked' => true,
                ]
            ]);

            return $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            Log::error('Channel Manager - Block Dates Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Map external room ID to local room
     */
    protected function mapExternalRoom($externalRoomId)
    {
        if (!$externalRoomId) {
            return null;
        }

        // Check if mapping exists in database
        $mapping = \DB::table('external_room_mappings')
            ->where('external_id', $externalRoomId)
            ->where('platform', 'channel_manager')
            ->first();

        if ($mapping) {
            return Room::find($mapping->room_id);
        }

        return null;
    }

    /**
     * Get external room ID from local room
     */
    protected function getExternalRoomId($roomId)
    {
        $mapping = \DB::table('external_room_mappings')
            ->where('room_id', $roomId)
            ->where('platform', 'channel_manager')
            ->first();

        return $mapping ? $mapping->external_id : null;
    }

    /**
     * Map external status to local status
     */
    protected function mapStatus($externalStatus)
    {
        $statusMap = [
            'confirmed' => 'confirmed',
            'checked_in' => 'confirmed',
            'checked_out' => 'completed',
            'canceled' => 'cancelled',
            'no_show' => 'cancelled',
        ];

        return $statusMap[strtolower($externalStatus)] ?? 'pending';
    }

    /**
     * Test API connection
     */
    public function testConnection()
    {
        try {
            $response = $this->client->get($this->baseUrl . 'getHotelDetails', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $this->apiKey,
                ],
                'query' => [
                    'propertyID' => $this->propertyId,
                ]
            ]);

            return $response->getStatusCode() === 200;
        } catch (\Exception $e) {
            Log::error('Channel Manager - Connection Test Failed: ' . $e->getMessage());
            return false;
        }
    }
}
