<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Room;

class RoomController extends Controller
{
    public function index()
    {
        $rooms = Room::all();
        return view('admin.rooms.index', compact('rooms'));
    }

    public function create()
    {
        $amenities = \App\Models\Amenity::all();
        $facilities = \App\Models\Facility::all();
        return view('admin.rooms.create', compact('amenities', 'facilities'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|string',
            'price' => 'required|numeric',
            'capacity' => 'required|integer',
            'description' => 'nullable|string',
            'is_available' => 'boolean',
            'image' => 'nullable|image|max:2048',
            'gallery_images' => 'nullable|array|max:5',
            'gallery_images.*' => 'image|max:2048',
            'amenities' => 'nullable|array',
            'facilities' => 'nullable|array',
        ]);

        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('rooms', 'public');
            $validated['image'] = $path;
        }

        $room = Room::create($validated);

        // Sync Amenities and Facilities
        if ($request->has('amenities')) {
            $room->amenities()->sync($request->amenities);
        }
        if ($request->has('facilities')) {
            $room->facilities()->sync($request->facilities);
        }

        // Handle Gallery Images
        if ($request->hasFile('gallery_images')) {
            foreach ($request->file('gallery_images') as $image) {
                $path = $image->store('room_gallery', 'public');
                $room->images()->create(['image_path' => $path]);
            }
        }

        return redirect()->route('admin.rooms.index')->with('success', 'Room created successfully.');
    }

    public function edit(Room $room)
    {
        $room->load(['amenities', 'facilities', 'images']);
        $amenities = \App\Models\Amenity::all();
        $facilities = \App\Models\Facility::all();
        return view('admin.rooms.edit', compact('room', 'amenities', 'facilities'));
    }

    public function update(Request $request, Room $room)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|string',
            'price' => 'required|numeric',
            'capacity' => 'required|integer',
            'description' => 'nullable|string',
            'is_available' => 'boolean',
            'image' => 'nullable|image|max:2048',
            'gallery_images' => 'nullable|array|max:5',
            'gallery_images.*' => 'image|max:2048',
            'amenities' => 'nullable|array',
            'facilities' => 'nullable|array',
            'remove_gallery_images' => 'nullable|array',
        ]);

        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('rooms', 'public');
            $validated['image'] = $path;
        }

        $validated['is_available'] = $request->has('is_available');
        $room->update($validated);

        // Sync Amenities and Facilities
        $room->amenities()->sync($request->amenities ?? []);
        $room->facilities()->sync($request->facilities ?? []);

        // Remove selected gallery images
        if ($request->has('remove_gallery_images')) {
            \App\Models\RoomImage::whereIn('id', $request->remove_gallery_images)->delete();
        }

        // Handle New Gallery Images
        if ($request->hasFile('gallery_images')) {
            foreach ($request->file('gallery_images') as $image) {
                $path = $image->store('room_gallery', 'public');
                $room->images()->create(['image_path' => $path]);
            }
        }

        return redirect()->route('admin.rooms.index')->with('success', 'Room updated successfully.');
    }

    public function destroy(Room $room)
    {
        $room->delete();
        return redirect()->route('admin.rooms.index')->with('success', 'Room deleted successfully.');
    }
}
