<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\Room;
use App\Models\Setting;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        // Get dashboard statistics
        $stats = [
            'total_rooms' => Room::count(),
            'total_bookings' => Booking::count(),
            'active_bookings' => Booking::where('status', 'confirmed')
                ->whereDate('check_in', '<=', now())
                ->whereDate('check_out', '>=', now())
                ->count(),
            'total_revenue' => Booking::where('status', 'confirmed')->sum('total_price'),
        ];

        // Get recent bookings (latest 5)
        $recent_bookings = Booking::with(['user', 'room'])
            ->latest()
            ->take(5)
            ->get();

        // Get room statistics
        $total_rooms = Room::count();
        
        // Occupied rooms = rooms with active confirmed bookings
        $occupied_rooms = Room::whereHas('bookings', function ($query) {
            $query->where('status', 'confirmed')
                ->whereDate('check_in', '<=', now())
                ->whereDate('check_out', '>=', now());
        })->count();
        
        // Unavailable rooms (is_available = false, can be used as "maintenance")
        $unavailable_rooms = Room::where('is_available', false)->count();
        
        // Available = total - occupied - unavailable
        $available_rooms = $total_rooms - $occupied_rooms - $unavailable_rooms;

        $room_stats = [
            'total' => $total_rooms,
            'available' => max(0, $available_rooms),
            'occupied' => $occupied_rooms,
            'maintenance' => $unavailable_rooms,
        ];

        // Get snow mode setting
        $snow_mode = Setting::where('key', 'snow_mode')->first()?->value ?? 'off';

        return view('admin.dashboard', compact(
            'stats',
            'recent_bookings',
            'room_stats',
            'snow_mode'
        ));
    }
}
