<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\ChannelManagerService;

class SyncChannelManagerBookings extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'channel:sync {--days=90 : Number of days to sync}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync bookings from channel manager (Booking.com, Airbnb, etc.)';

    /**
     * Execute the console command.
     */
    public function handle(ChannelManagerService $channelManager)
    {
        $this->info('🔄 Starting channel manager sync...');
        
        // Test connection first
        if (!$channelManager->testConnection()) {
            $this->error('❌ Failed to connect to channel manager API');
            $this->error('Please check your API credentials in .env file');
            return 1;
        }

        $this->info('✅ Connected to channel manager');

        // Calculate date range
        $days = $this->option('days');
        $startDate = now()->format('Y-m-d');
        $endDate = now()->addDays($days)->format('Y-m-d');

        $this->info("📅 Syncing reservations from {$startDate} to {$endDate}");

        // Sync reservations
        $result = $channelManager->getReservations($startDate, $endDate);

        if ($result && isset($result['data'])) {
            $count = count($result['data']);
            $this->info("✅ Successfully synced {$count} reservations");
            
            // Show summary
            $this->table(
                ['Source', 'Guest', 'Check-in', 'Check-out', 'Status'],
                collect($result['data'])->take(10)->map(function($reservation) {
                    return [
                        $reservation['source'] ?? 'Unknown',
                        $reservation['guestName'] ?? 'N/A',
                        $reservation['startDate'] ?? 'N/A',
                        $reservation['endDate'] ?? 'N/A',
                        $reservation['status'] ?? 'N/A',
                    ];
                })->toArray()
            );

            if ($count > 10) {
                $this->info("... and " . ($count - 10) . " more");
            }
        } else {
            $this->warn('⚠️  No reservations found or sync failed');
        }

        $this->info('✅ Sync completed!');
        return 0;
    }
}
