# Integration Guide - Booking.com & Airbnb APIs

This guide explains how to integrate Booking.com and Airbnb APIs into your hotel booking system.

---

## Overview

Both platforms offer different integration methods:
- **Booking.com**: XML/API integration via their Connectivity Partner Program
- **Airbnb**: API integration for professional hosts and property managers

---

## 1. Booking.com Integration

### Step 1: Register as a Connectivity Partner

1. Visit [Booking.com Connectivity](https://connect.booking.com/)
2. Apply to become a Connectivity Partner
3. Wait for approval (can take several weeks)
4. Receive API credentials

### Step 2: API Access Types

**Option A: XML API (Most Common)**
- Real-time availability updates
- Rate and inventory management
- Reservation delivery
- Modification handling

**Option B: Booking.com API v2**
- RESTful API
- More modern approach
- Better documentation

### Step 3: Implementation

```php
// Install Guzzle HTTP client
composer require guzzlehttp/guzzle

// Create Booking.com Service
// app/Services/BookingComService.php
<?php

namespace App\Services;

use GuzzleHttp\Client;

class BookingComService
{
    protected $client;
    protected $hotelId;
    protected $username;
    protected $password;

    public function __construct()
    {
        $this->client = new Client([
            'base_uri' => 'https://supply-xml.booking.com/hotels/xml/',
            'timeout' => 30,
        ]);
        
        $this->hotelId = config('services.booking.hotel_id');
        $this->username = config('services.booking.username');
        $this->password = config('services.booking.password');
    }

    // Push room availability
    public function updateAvailability($roomId, $date, $available)
    {
        $xml = $this->buildAvailabilityXML($roomId, $date, $available);
        
        return $this->client->post('push', [
            'auth' => [$this->username, $this->password],
            'body' => $xml,
            'headers' => ['Content-Type' => 'application/xml']
        ]);
    }

    // Push room rates
    public function updateRates($roomId, $date, $price)
    {
        $xml = $this->buildRatesXML($roomId, $date, $price);
        
        return $this->client->post('push', [
            'auth' => [$this->username, $this->password],
            'body' => $xml,
            'headers' => ['Content-Type' => 'application/xml']
        ]);
    }

    // Receive reservations
    public function receiveReservation($xmlData)
    {
        // Parse XML reservation data
        $reservation = simplexml_load_string($xmlData);
        
        // Create booking in your system
        $booking = Booking::create([
            'external_id' => (string)$reservation->reservation_id,
            'source' => 'booking.com',
            'room_id' => $this->mapExternalRoom($reservation->room_id),
            'check_in' => (string)$reservation->checkin,
            'check_out' => (string)$reservation->checkout,
            'guest_name' => (string)$reservation->customer->name,
            'total_price' => (float)$reservation->total_price,
        ]);

        return $booking;
    }

    protected function buildAvailabilityXML($roomId, $date, $available)
    {
        return <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<request>
    <username>{$this->username}</username>
    <password>{$this->password}</password>
    <hotel_id>{$this->hotelId}</hotel_id>
    <room_id>{$roomId}</room_id>
    <date>{$date}</date>
    <availability>{$available}</availability>
</request>
XML;
    }
}
```

### Step 4: Configuration

```php
// config/services.php
'booking' => [
    'hotel_id' => env('BOOKING_HOTEL_ID'),
    'username' => env('BOOKING_USERNAME'),
    'password' => env('BOOKING_PASSWORD'),
],
```

```env
# .env
BOOKING_HOTEL_ID=your_hotel_id
BOOKING_USERNAME=your_username
BOOKING_PASSWORD=your_password
```

### Step 5: Webhook Endpoint

```php
// routes/api.php
Route::post('/webhooks/booking-com', [BookingComWebhookController::class, 'handle']);

// app/Http/Controllers/BookingComWebhookController.php
<?php

namespace App\Http\Controllers;

use App\Services\BookingComService;
use Illuminate\Http\Request;

class BookingComWebhookController extends Controller
{
    public function handle(Request $request, BookingComService $service)
    {
        $xmlData = $request->getContent();
        
        try {
            $booking = $service->receiveReservation($xmlData);
            
            return response()->xml([
                'status' => 'OK',
                'reservation_id' => $booking->external_id
            ]);
        } catch (\Exception $e) {
            return response()->xml([
                'status' => 'ERROR',
                'message' => $e->getMessage()
            ], 400);
        }
    }
}
```

---

## 2. Airbnb Integration

### Step 1: Join Airbnb Partner Program

1. Visit [Airbnb for Partners](https://www.airbnb.com/partners)
2. Apply as a Software Partner
3. Complete verification process
4. Receive API credentials

### Step 2: OAuth Authentication

```php
// Install Laravel Socialite
composer require laravel/socialite

// config/services.php
'airbnb' => [
    'client_id' => env('AIRBNB_CLIENT_ID'),
    'client_secret' => env('AIRBNB_CLIENT_SECRET'),
    'redirect' => env('AIRBNB_REDIRECT_URI'),
],
```

### Step 3: Airbnb Service

```php
// app/Services/AirbnbService.php
<?php

namespace App\Services;

use GuzzleHttp\Client;

class AirbnbService
{
    protected $client;
    protected $accessToken;

    public function __construct()
    {
        $this->client = new Client([
            'base_uri' => 'https://api.airbnb.com/v2/',
            'timeout' => 30,
        ]);
        
        $this->accessToken = config('services.airbnb.access_token');
    }

    // Get listings
    public function getListings()
    {
        $response = $this->client->get('listings', [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->accessToken,
                'Accept' => 'application/json',
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    // Update calendar (availability)
    public function updateCalendar($listingId, $date, $available, $price = null)
    {
        $data = [
            'listing_id' => $listingId,
            'start_date' => $date,
            'end_date' => $date,
            'availability' => $available ? 'available' : 'unavailable',
        ];

        if ($price) {
            $data['daily_price'] = $price;
        }

        return $this->client->put("calendar_days/{$listingId}", [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->accessToken,
                'Content-Type' => 'application/json',
            ],
            'json' => $data
        ]);
    }

    // Get reservations
    public function getReservations($listingId = null)
    {
        $params = ['status' => 'accepted'];
        if ($listingId) {
            $params['listing_id'] = $listingId;
        }

        $response = $this->client->get('reservations', [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->accessToken,
            ],
            'query' => $params
        ]);

        return json_decode($response->getBody(), true);
    }

    // Sync reservation to local database
    public function syncReservation($airbnbReservation)
    {
        return Booking::updateOrCreate(
            ['external_id' => $airbnbReservation['id']],
            [
                'source' => 'airbnb',
                'room_id' => $this->mapAirbnbListing($airbnbReservation['listing_id']),
                'check_in' => $airbnbReservation['start_date'],
                'check_out' => $airbnbReservation['end_date'],
                'guest_name' => $airbnbReservation['guest']['name'],
                'total_price' => $airbnbReservation['total_price'],
                'status' => $airbnbReservation['status'],
            ]
        );
    }

    protected function mapAirbnbListing($airbnbListingId)
    {
        // Map Airbnb listing ID to your room ID
        $mapping = AirbnbRoomMapping::where('airbnb_listing_id', $airbnbListingId)->first();
        return $mapping ? $mapping->room_id : null;
    }
}
```

### Step 4: Webhook Handler

```php
// routes/api.php
Route::post('/webhooks/airbnb', [AirbnbWebhookController::class, 'handle']);

// app/Http/Controllers/AirbnbWebhookController.php
<?php

namespace App\Http\Controllers;

use App\Services\AirbnbService;
use Illuminate\Http\Request;

class AirbnbWebhookController extends Controller
{
    public function handle(Request $request, AirbnbService $service)
    {
        $event = $request->input('event_type');
        
        switch ($event) {
            case 'reservation.created':
            case 'reservation.updated':
                $reservation = $request->input('reservation');
                $service->syncReservation($reservation);
                break;
                
            case 'reservation.cancelled':
                $this->handleCancellation($request->input('reservation'));
                break;
        }

        return response()->json(['status' => 'success']);
    }

    protected function handleCancellation($reservation)
    {
        Booking::where('external_id', $reservation['id'])
            ->update(['status' => 'cancelled']);
    }
}
```

---

## 3. Channel Manager Approach (Recommended)

Instead of direct integration, consider using a **Channel Manager** service:

### Popular Channel Managers:
1. **Cloudbeds** - All-in-one PMS + Channel Manager
2. **SiteMinder** - Global channel manager
3. **RoomRaccoon** - Affordable option
4. **Beds24** - Budget-friendly

### Benefits:
- ✅ Single integration point
- ✅ Manage multiple OTAs (Booking.com, Airbnb, Expedia, etc.)
- ✅ Automatic 2-way sync
- ✅ Prevent overbooking
- ✅ Centralized calendar
- ✅ Rate management

### Example: Cloudbeds Integration

```php
// app/Services/CloudbedsService.php
<?php

namespace App\Services;

use GuzzleHttp\Client;

class CloudbedsService
{
    protected $client;
    protected $propertyId;
    protected $apiKey;

    public function __construct()
    {
        $this->client = new Client([
            'base_uri' => 'https://api.cloudbeds.com/api/v1.1/',
        ]);
        
        $this->propertyId = config('services.cloudbeds.property_id');
        $this->apiKey = config('services.cloudbeds.api_key');
    }

    // Get all reservations
    public function getReservations($startDate, $endDate)
    {
        $response = $this->client->get('getReservations', [
            'headers' => ['Authorization' => 'Bearer ' . $this->apiKey],
            'query' => [
                'propertyID' => $this->propertyId,
                'startDate' => $startDate,
                'endDate' => $endDate,
            ]
        ]);

        return json_decode($response->getBody(), true);
    }

    // Update room availability
    public function updateAvailability($roomId, $date, $quantity)
    {
        return $this->client->post('putRoomAvailability', [
            'headers' => ['Authorization' => 'Bearer ' . $this->apiKey],
            'json' => [
                'propertyID' => $this->propertyId,
                'roomTypeID' => $roomId,
                'date' => $date,
                'quantity' => $quantity,
            ]
        ]);
    }
}
```

---

## 4. Database Schema Updates

```php
// Create migration for external mappings
php artisan make:migration create_external_room_mappings_table

// Migration
Schema::create('external_room_mappings', function (Blueprint $table) {
    $table->id();
    $table->foreignId('room_id')->constrained()->onDelete('cascade');
    $table->string('platform'); // 'booking.com', 'airbnb', etc.
    $table->string('external_id'); // Platform's room/listing ID
    $table->json('sync_settings')->nullable();
    $table->timestamps();
    
    $table->unique(['room_id', 'platform']);
});

// Update bookings table
Schema::table('bookings', function (Blueprint $table) {
    $table->string('source')->default('direct'); // 'direct', 'booking.com', 'airbnb'
    $table->string('external_id')->nullable();
    $table->json('external_data')->nullable();
});
```

---

## 5. Sync Strategy

```php
// app/Console/Commands/SyncExternalBookings.php
<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\BookingComService;
use App\Services\AirbnbService;

class SyncExternalBookings extends Command
{
    protected $signature = 'bookings:sync';
    protected $description = 'Sync bookings from external platforms';

    public function handle(
        BookingComService $bookingCom,
        AirbnbService $airbnb
    ) {
        $this->info('Syncing Booking.com reservations...');
        // Sync Booking.com
        
        $this->info('Syncing Airbnb reservations...');
        $reservations = $airbnb->getReservations();
        foreach ($reservations['reservations'] as $reservation) {
            $airbnb->syncReservation($reservation);
        }
        
        $this->info('Sync completed!');
    }
}

// Schedule in app/Console/Kernel.php
protected function schedule(Schedule $schedule)
{
    $schedule->command('bookings:sync')->everyFifteenMinutes();
}
```

---

## 6. Implementation Checklist

- [ ] Choose integration method (Direct API vs Channel Manager)
- [ ] Register with Booking.com/Airbnb partner programs
- [ ] Obtain API credentials
- [ ] Set up database schema for external mappings
- [ ] Create service classes for each platform
- [ ] Implement webhook endpoints
- [ ] Build admin UI for room mapping
- [ ] Set up automated sync (cron jobs)
- [ ] Test reservation flow end-to-end
- [ ] Monitor for sync errors
- [ ] Implement error handling and logging

---

## Recommendation

For your use case, I recommend:

1. **Start with a Channel Manager** (like Cloudbeds or SiteMinder)
   - Faster implementation
   - Less maintenance
   - Better reliability

2. **If you need direct integration:**
   - Start with Booking.com (larger market share)
   - Add Airbnb later
   - Use webhooks for real-time updates

3. **Essential features to implement:**
   - Two-way calendar sync
   - Automatic rate updates
   - Overbooking prevention
   - Centralized reservation management

Would you like me to implement any specific part of this integration?
