# Channel Manager Integration - Setup Guide

## ✅ What's Been Implemented

I've set up a complete Channel Manager integration system for your hotel booking application. This allows you to sync bookings from Booking.com, Airbnb, and other OTAs through a single integration point.

---

## 📦 Components Created

### 1. **ChannelManagerService** (`app/Services/ChannelManagerService.php`)
Main service for interacting with the channel manager API:
- ✅ Sync reservations from all platforms
- ✅ Update room availability across all channels
- ✅ Update room rates
- ✅ Block dates
- ✅ Automatic mapping between external and local rooms

### 2. **Database Migration** (`database/migrations/*_create_external_room_mappings_table.php`)
- ✅ `external_room_mappings` table - Maps your rooms to channel manager room IDs
- ✅ Updated `bookings` table with:
  - `source` - Track booking source (direct, booking.com, airbnb, etc.)
  - `external_id` - External reservation ID
  - `external_data` - Full external booking data (JSON)
  - `guest_name`, `guest_email`, `guest_phone` - For external bookings

### 3. **Sync Command** (`app/Console/Commands/SyncChannelManagerBookings.php`)
Automated command to sync bookings:
```bash
php artisan channel:sync
php artisan channel:sync --days=30
```

### 4. **Configuration** (`config/services.php`)
Centralized configuration for all channel manager settings

---

## 🚀 Setup Instructions

### Step 1: Choose Your Channel Manager

**Recommended Options:**
1. **Cloudbeds** - Most popular, all-in-one
2. **SiteMinder** - Global reach
3. **Beds24** - Budget-friendly

### Step 2: Sign Up & Get API Credentials

1. Visit your chosen channel manager's website
2. Sign up for an account
3. Connect your property
4. Get API credentials (Property ID, API Key, API Secret)

### Step 3: Configure Environment Variables

Add to your `.env` file:

```env
# Channel Manager Configuration
CHANNEL_MANAGER_PROVIDER=cloudbeds
CHANNEL_MANAGER_PROPERTY_ID=your_property_id_here
CHANNEL_MANAGER_API_KEY=your_api_key_here
CHANNEL_MANAGER_API_SECRET=your_api_secret_here
CHANNEL_MANAGER_BASE_URL=https://api.cloudbeds.com/api/v1.1/

# Sync Settings
CHANNEL_MANAGER_AUTO_SYNC=true
CHANNEL_MANAGER_SYNC_INTERVAL=15
```

### Step 4: Run Database Migration

```bash
php artisan migrate
```

This creates:
- `external_room_mappings` table
- Updates `bookings` table with new fields

### Step 5: Map Your Rooms

You need to map your local rooms to the channel manager's room IDs:

```php
// In tinker or a seeder
DB::table('external_room_mappings')->insert([
    'room_id' => 1, // Your local room ID
    'platform' => 'channel_manager',
    'external_id' => 'CM_ROOM_123', // Channel manager's room ID
    'auto_sync' => true,
    'created_at' => now(),
    'updated_at' => now(),
]);
```

### Step 6: Test the Connection

```bash
php artisan tinker
```

```php
$cm = app(\App\Services\ChannelManagerService::class);
$cm->testConnection(); // Should return true
```

### Step 7: Sync Bookings

```bash
# Sync next 90 days
php artisan channel:sync

# Sync next 30 days
php artisan channel:sync --days=30
```

### Step 8: Schedule Automatic Sync

Add to `app/Console/Kernel.php`:

```php
protected function schedule(Schedule $schedule)
{
    // Sync every 15 minutes
    $schedule->command('channel:sync')->everyFifteenMinutes();
}
```

---

## 💡 How It Works

### Booking Flow:

1. **Guest books on Booking.com/Airbnb** →
2. **Channel Manager receives booking** →
3. **Your system syncs via API** (every 15 min or real-time webhook) →
4. **Booking appears in your dashboard** with source tag
5. **Availability automatically updated** across all platforms

### Availability Updates:

```php
use App\Services\ChannelManagerService;

$cm = app(ChannelManagerService::class);

// Update availability
$cm->updateAvailability(
    roomId: 1,
    date: '2025-01-15',
    quantity: 5 // 5 rooms available
);

// Update rates
$cm->updateRates(
    roomId: 1,
    date: '2025-01-15',
    price: 15000 // LKR 15,000
);

// Block dates
$cm->blockDates(
    roomId: 1,
    startDate: '2025-01-20',
    endDate: '2025-01-25'
);
```

---

## 🎯 Next Steps

### 1. Create Admin UI for Room Mapping
Create an admin page where you can:
- View all rooms
- Map/unmap rooms to channel manager
- See sync status

### 2. Add Webhook Support (Real-time)
Instead of polling every 15 minutes, receive instant notifications:
- Create webhook endpoint
- Register with channel manager
- Process bookings in real-time

### 3. Add Error Monitoring
- Log all sync errors
- Email notifications for failed syncs
- Dashboard for sync health

### 4. Implement Rate Management
- Bulk rate updates
- Seasonal pricing
- Last-minute deals

---

## 📊 Benefits

✅ **Single Integration** - Manage all OTAs from one place  
✅ **No Overbooking** - Automatic availability sync  
✅ **Centralized Calendar** - One calendar for all platforms  
✅ **Time Saving** - No manual updates needed  
✅ **More Bookings** - Reach customers on multiple platforms  
✅ **Professional** - Compete with major hotels  

---

## 🆘 Troubleshooting

**Connection Failed?**
- Check API credentials in `.env`
- Verify property ID is correct
- Ensure API key has proper permissions

**Rooms Not Syncing?**
- Check room mappings in `external_room_mappings` table
- Verify external room IDs match channel manager

**Bookings Not Appearing?**
- Run `php artisan channel:sync` manually
- Check logs in `storage/logs/laravel.log`
- Verify date range

---

## 📞 Support

Need help? Contact your channel manager's support:
- **Cloudbeds**: support@cloudbeds.com
- **SiteMinder**: support@siteminder.com
- **Beds24**: support@beds24.com

---

Ready to connect to Booking.com, Airbnb, and 100+ other booking platforms! 🚀
